% model the sun's spectrum with blackbody curve
% ASTM G173-03 Reference Spectrum Etv in W/m^2 per nm 
% first column is wavelength in nm, second is intensity (W/m^2 per nm)
a=xlsread('sun.xlsx'); 
% convert to SI units
a(:,1)=a(:,1)*1e-9; % convert nm to m

% model with blackbody curve
h=6.6260755e-34; % J s, Planck's constant
c=2.99792458e8; % m/s, speed of light in vacuum
k=1.380658e-23; % J/K, Boltmann's constant
lam=a(:,1);
T=5780; % temperature in K
p=((8*pi*h*c)./lam.^5)./(exp(h*c./(k*lam.*T))-1);

% Define and Perform Nonlinear Least Squares Fit at Long Wavelengths
x=a(1041:2002,1); % all above 1.2 microns
y=a(1041:2002,2);  % all above 1.2 microns
b0=[0.2e-5] % choose initial fit parameter
fun=@(b,x) b(1)*((8*pi*h*c)./x.^5)./(exp(h*c./(k*x.*T))-1); % define fit function
mdl = NonLinearModel.fit(x,y,fun,b0); % MATLAB toolbox function
mdl % report results in command window
yfit=mdl.Fitted; % fitted values of y
bfit=mdl.Coefficients.Estimate; % fit parameters
bfitSE=mdl.Coefficients.SE; % fit parameter uncertainties
yfit2=bfit*p;  % fit over whole range, not just long wavelengths
sun_abs=-log10(a(:,2)./yfit2);  % absorption spectrum of surface

% fit of scaling with fixed temperature
figure(2)
plot(a(:,1),a(:,2),'g')
hold on
plot(a(:,1),yfit2,'r','LineWidth',2)
hold on
plot(x,yfit,'Color',[1 .6 0],'LineWidth',2)
set(gca,'Fontsize',12)
xlabel('Wavelength (m)','FontSize',14)
ylabel('E Flux/Bandwidth (W m^-2 nm^-1)','FontSize',12)
text(1e-6,1.2,[num2str(T),' K'],'FontSize',14,'Color',[1 0 0])
set(gcf,'PaperUnits','inches','PaperPosition',[0 0 6 3])
print(2,'-dtiff','-r600','blackbody_sun_1.tif')

