% DVRgeneral, Discrete Variable Representation, Finite Well
% J.V.Coe, 8/23/2012, modified 1/8/2018
% D.T. Colbert and W.H. Miller, J. Chem. Phys. 96, 1982-91 (1992).
% All atomic units
clear; clf; % clear variables and plots
hbar=1; % hbar in atomic units
mu=1; % mass in electron masses 
ps=0.2; % pick a wavefunction amplitude scaling factor
nE=4; % pick the number of energy levels to plot
% ********** define grid of independent variable ********************
imax=1000; % number of steps along independent coordinate
xmin=-20; % min value of independent coordinate
xmax=20; % max value of independent coordinate
dx=(xmax-xmin)/(imax-1); % increment, step size 
x=xmin:dx:xmax; % define grid of x-variable, start:step:end
gp=hbar^2/(2*mu*dx^2); % calculate grid parameter

% input Well Parameters from Dialog Text Box
disp('particle is an electron')
disp('a=Left Edge of Well 1 (bohr)')
disp('b=Right Edge of Well 1 (bohr)')
disp('c=Left Edge of Well 2 (bohr)')
disp('d=Right Edge of Well 2 (bohr)')
disp('D=Well Depth (Hartree)')
disp('Upper Max of Plot')
prompt = {'a left edge Well 1 (-20 to 20) :', ...
    'b right edge Well 1 (>a, -20 to 20):', ...
    'c left edge Well 2 (-20 to 20) :', ...
    'd right edge Well 2 (>a, -20 to 20):', ...
    'Well Depth (Hartree) :',...
    'Max of Energy Scale (Hartree)'};
dlgtitle = 'Input';
fieldsize = [1 20; 1 20; 1 20; 1 20; 1 20; 1 20];
definput = {'-5','-0.5','0.5','5','0.5','0.7'};
answer = inputdlg(prompt,dlgtitle,fieldsize,definput);
edge1 = str2num(answer{1});
edge2 = str2num(answer{2});
edge3 = str2num(answer{3});
edge4 = str2num(answer{4});
D = str2num(answer{5});
Escale = str2num(answer{6});

% ********** get potential function **********************************
for i=1:imax
    V(i)=D; % potential outside of box
    if  (edge1<=x(i)&&x(i)<=edge2) || (edge3<=x(i)&&x(i)<=edge4) 
        V(i)=0; % potential inside of box
    end
end
% ********** kinetic energy matrix ************************************
T=zeros(imax,imax);
for i=1:imax
    for ip=1:imax
        T(i,ip)=gp*(pi^2)/3.0;
        if(ip~=i); T(i,ip)=gp*((-1)^(i-ip))*2/((i-ip)^2); end
    end
end
% **********  potential energy matrix ********************************
Vh=zeros(imax,imax);
for i=1:imax
    for ip=1:imax; if(i==ip); Vh(i,ip)=V(i); end; end
end
% ********** Hamiltonian *********************************************
H=T+Vh;
% ********** Digonalize Hamiltonian **********************************
% Calc. eigenvalues (on diagaonal of E1) and eigenvectors as columns of P
[P,E1]=eig(H);
E=diag(E1); % Pick off the eigenvalues
% ********** Normalization of Wavefunctions *************************
N=0;
for i=1:imax; N=N+conj(P(i,1))*P(i,1)*dx; end
disp(['check for normalization, N= ',num2str(N)])
P=P./(N^0.5); % normalize wavefunctions
% check for normalization
s=0;
for i=1:imax; s=s+conj(P(i,1))*P(i,1)*dx; end
disp(['after nomalization, N= ',num2str(s)])
% ********** Plot the results ***************************************
for n=1:nE % choose the number of states to plot 
    plot(x(:),V(:),'r'); hold on
    plot(x(:),E(n,1)*ones(imax),'-g'); hold on % green energy level lines
    plot(x(:),real(P(:,n))*ps+E(n,1),'b'); hold on % modify wavefunc. amplitude
    text(-19,E(n,1),['E(',num2str(n),')= ',num2str(E(n,1))],'Color',[0 0 0]);
end
axis([-20 20 -0.1 Escale]) % scale the plot
set(gca,'FontSize',12)
xlabel('x (bohr)','FontSize',14,'Interpreter','latex')
ylabel('\color{green}E (green, Hartree), \color{blue} \psi (blue, scaled)')
set(gcf,'PaperUnits','inches','PaperPosition',[0 0 5 5])
print(1,'-dtiff','-r600','DVR_2_wells.jpg') 

