% DVRgeneral, Discrete Variable Representation, Finite Well
% J.V.Coe, 8/23/2012, modified 1/8/2018, 9/27/2025
% D.T. Colbert and W.H. Miller, J. Chem. Phys. 96, 1982-91 (1992).
% All atomic units
clear; clf; % clear variables and figures
%************************** INPUT ****************************************
hbar=1; % hbar in atomic units
mu=1; % mass in electron masses 
% Dialog Box Input
disp('Electron in a Finite Well')
disp('a Left Edge, b Right Edge of box (bohr)')
disp('Well Depth (Hartree)')
prompt = {'Enter a left edge (-20 to 20) :', ...
  'Enter b right edge (>a, -20 to 20):','Enter Well Depth (Hartree) :',...
  '# states to plot ','wavefunction amplitude scale','plot Emax'};
dlgtitle = 'Input';
fieldsize = [1 20; 1 20; 1 20; 1 20; 1 20; 1 20];
definput = {'0','10','1','5','0.2','1.2'};
answer = inputdlg(prompt,dlgtitle,fieldsize,definput);
edge1 = str2num(answer{1});
edge2 = str2num(answer{2});
D = str2num(answer{3});
nE = str2num(answer{4}); % pick the number of energy levels to plot
ps = str2num(answer{5}); % pick a wavefunction amplitude scaling factor
Emax = str2num(answer{6}); % pick a wavefunction amplitude scaling factor
%************************************************************************* 

%  ********** define grid of independent variable ************************
imax=1000; % number of steps along independent coordinate
xmin=-20; % min value of independent coordinate
xmax=20; % max value of independent coordinate
dx=(xmax-xmin)/(imax-1); % increment, step size 
x=xmin:dx:xmax; % define grid of x-variable, start:step:end
gp=hbar^2/(2*mu*dx^2); % calculate grid parameter
% ********** input the potential function *****************************
V=zeros(imax); % initial potential
for i=1:imax
    V(i)=D; % potential outside of box
    if(edge1<=x(i)&&x(i)<=edge2); V(i)=0; end % potential inside of box
end

% ********** kinetic energy matrix ************************************
T=zeros(imax,imax);
for i=1:imax
    for ip=1:imax
        T(i,ip)=gp*(pi^2)/3.0;
        if(ip~=i); T(i,ip)=gp*((-1)^(i-ip))*2/((i-ip)^2); end
    end
end
% **********  potential energy matrix ********************************
Vh=zeros(imax,imax);
for i=1:imax
    for ip=1:imax
        if(i==ip); Vh(i,ip)=V(i); end
    end
end
% ********** Hamiltonian *********************************************
H=T+Vh;
% ********** Diagonalize Hamiltonian **********************************
% Calculate eigenvalues (on diagonal of E1) and
% eigenvectors as columns of P
[P,E1]=eig(H);
% Pick off the eigenvalues
E=diag(E1);
% ********** Normalization of Wavefunctions *************************
N=0;
for i=1:imax; N=N+conj(P(i,1))*P(i,1)*dx; end
disp(['check for normalization, N= ',num2str(N)])
% normalize wavefunctions
P=P./(N^0.5);
% check for normalization
s=0;
for i=1:imax; s=s+conj(P(i,1))*P(i,1)*dx; end
disp(['after nomalization, N= ',num2str(s)])
 
% ********** Plot the results ***************************************
figure(1)
set(text,'Color','g')
for n=1:nE % choose the number of states to plot 
    plot(x(:),V(:),'r'); hold on
    plot(x(:),E(n,1)*ones(imax),'-g'); hold on % green energy level lines
    text(-19,E(n,1),['E(',num2str(n),')= ',num2str(E(n,1))],'Color',[0 0 0]);
    plot(x(:),P(:,n)*ps+E(n,1),'b'); hold on % can multiply P by -1 for phase
end
axis([-20 20 -0.1 Emax]) % scale the plot
set(gca,'FontSize',12)
xlabel('x (bohr)','FontSize',14,'Interpreter','latex')
ylabel('\color{green}E (green, Hartree), \color{blue} \psi (blue, scaled)')
set(gcf,'PaperUnits','inches','PaperPosition',[0 0 5 5])
print(1,'-dtiff','-r600','DVR.jpg') 

